﻿using System;
using System.Collections;
using System.Collections.Generic;
using System.Linq;
using TMPro;
using UnityEngine;
using UnityEngine.UI;

//Settings is a script for managing the settings page UI
public class Settings : MonoBehaviour
{
    private Resolution[] _Resolutions;

    //UI Elements
    public TMP_Dropdown ResolutionsDropdown;
    public Toggle FullScreenToggle;
    public TMP_Dropdown QualitiesDropdown;
    public Slider MusicSlider;
    public Slider SFXSlider;
    public Slider VehicleVolume;

    private void Start()
    {
        //Set up the UI
        FullScreenToggle.isOn = Screen.fullScreen;
        MusicSlider.value = Convert.ToSingle(SettingsManager.Instance.Settings[Constants.Settings.MusicVolume]);
        SFXSlider.value = Convert.ToSingle(SettingsManager.Instance.Settings[Constants.Settings.SFXVolume]);
        VehicleVolume.value = Convert.ToSingle(SettingsManager.Instance.Settings[Constants.Settings.VehicleVolume]);
        SetupResolutions();
        SetupQualities();
    }

    /// <summary>
    /// Sets up the resolution dropdown
    /// </summary>
    private void SetupResolutions()
    {
        _Resolutions = Screen.resolutions.Where(resolution => resolution.refreshRate == 60).ToArray();  //Only 60Hz resolutions to avoid duplicates
        ResolutionsDropdown.ClearOptions();

        List<string> options = new List<string>();

        int currentResIndex = 0;    //The index of the game's current resolution
        for (int i = 0; i < _Resolutions.Length; i++)
        {
            if (!options.Contains(_Resolutions[i].width + "x" + _Resolutions[i].height))
            {
                //No duplicates, add it
                options.Add(_Resolutions[i].width + "x" + _Resolutions[i].height);
            }

            if (_Resolutions[i].width == Screen.width && _Resolutions[i].height == Screen.height)
            {
                //Found our current resolution
                currentResIndex = options.Count - 1;
            }
        }

        ResolutionsDropdown.AddOptions(options);    //Add the resolutions

        //Set the selected value to be the current game resolution
        ResolutionsDropdown.value = currentResIndex;
        ResolutionsDropdown.RefreshShownValue();
    }

    /// <summary>
    /// Sets up the qualities dropdown
    /// </summary>
    private void SetupQualities()
    {
        QualitiesDropdown.ClearOptions();

        List<string> options = QualitySettings.names.ToList();  //Get all the qualities

        int currentResIndex = 0;
        for (int i = 0; i < QualitySettings.names.Length; i++)
        {
            if (QualitySettings.GetQualityLevel() == i)
            {
                //Found our current quality, break from the loop
                currentResIndex = i;
                break;
            }
        }

        QualitiesDropdown.AddOptions(options);  //Add the qualities

        //Set the selected value to be the current quality
        QualitiesDropdown.value = currentResIndex;
        QualitiesDropdown.RefreshShownValue();
    }

    /// <summary>
    /// Sets the resolution based on the index into the list of resolutions
    /// </summary>
    /// <param name="index">The index of the resolution to set</param>
    public void SetResolution(int index)
    {
        //Set the resolution based on the index
        Resolution res = _Resolutions[index];
        Screen.SetResolution(res.width, res.height, Screen.fullScreen);
    }

    /// <summary>
    /// Sets the quality based on the index into the list of qualities
    /// </summary>
    /// <param name="index">The index of the quality to set</param>
    public void SetQuality(int index)
    {
        //Set the quality based on the index
        QualitySettings.SetQualityLevel(index, true);
    }

    /// <summary>
    /// Sets the screen's full screen state
    /// </summary>
    /// <param name="fullScreen">Should we be in full screen?</param>
    public void SetFullScreen(bool fullScreen)
    {
        Screen.fullScreen = fullScreen;
    }

    /// <summary>
    /// Sets the volume of all music
    /// </summary>
    /// <param name="volume">The volume level to set</param>
    public void SetMusicVolume(float volume)
    {
        //Set the volume, pass true to overwrite
        AudioManager.Instance.SetMusicVolume(volume, true);
    }

    /// <summary>
    /// Sets the volume of all SFX
    /// </summary>
    /// <param name="volume">The volume level to set</param>
    public void SetSFXVolume(float volume)
    {
        //Set the volume, pass true to overwrite
        AudioManager.Instance.SetSFXVolume(volume, true);
    }

    /// <summary>
    /// Sets the volume of all vehicle audio
    /// </summary>
    /// <param name="volume">The volume level to set</param>
    public void SetVehicleVolume(float volume)
    {
        SettingsManager.Instance.Settings[Constants.Settings.VehicleVolume] = volume;
        SettingsManager.Instance.WriteSettings();
    }
}
